<?php

namespace Controller;

use Firebase\JWT\JWT;
use Inc\Helper;
use Inc\Email;
use Model\User;

class Auth extends Base {

  /**
   * Before a page is served
   * @param $f3
   * @param $params
   */
  public function beforeroute($f3, $params) {
    parent::beforeroute($f3, $params);
    $this->view->templateFolder = 'auth';

    // if user is already logged in redirect them accordingly
    $user = $this->user();
    if ($user && $f3->get('ALIAS') != 'logout') {
      $f3->reroute($user['admin'] ? '@backend_settings' : '@index');
    }
  }

  /**
   * After page is routed
   * @param $f3
   * @param $params
   */
  public function afterroute($f3, $params) {
    // set title and meta variables
    $alias = $f3->get('ALIAS');
    $this->view->data['page_title'] = $f3->get('site.title')[$alias];
    $this->view->data['page_meta_description'] = $f3->get('site.meta_description')[$alias];

    parent::afterroute($f3, $params);
  }

  /**
   * Login page
   * @param $f3
   * @param $params
   */
  public function login($f3, $params) {
    // if user tries to log in
    if ($f3->get('VERB')=='POST') {
      $user = new User();
      // if user email and password are correct
      if ($loggedUser = $user->login($f3->get('POST.email'), $f3->get('POST.password'))->get()) {
        $this->startSession($loggedUser);
        $destination = $f3->get('POST.destination');
        $f3->reroute($destination ? $destination : ($loggedUser['admin'] ? '@backend_settings' : '@index'));
      } else {
        $this->view->data['form'] = [
          'email' => $f3->get('POST.email'),
          'password' => $f3->get('POST.password'),
        ];
        $this->view->data['errors']['login'] = $f3->get('text.incorrect_login');
      }
    }
  }

  /**
   * Log user out
   * @param $f3
   * @param $params
   */
  public function logout($f3, $params) {
    $this->endSession();
    $f3->reroute('@index');
  }

  /**
   * Registration
   * @param $f3
   * @param $params
   */
  public function register($f3, $params) {
    // if user tries to register
    if ($f3->get('VERB')=='POST') {
      $user = new User();
      $firstName  = $f3->get('POST.first_name');
      $lastName   = $f3->get('POST.last_name');
      $email      = $f3->get('POST.email');
      $password   = $f3->get('POST.password');

      // check if user already exists
      if (!$user->count('email', $email)) {
        // try to create user
        if ($user->create([
          'first_name'  => $firstName,
          'last_name'   => $lastName,
          'email'       => $email,
          'password'    => $password,
          'admin'       => 0
        ])->lastId()) {
          $user->update(['last_login' => Helper::timeStamp()]);
          $this->startSession($user->get());
          $destination = $f3->get('POST.destination');
          $f3->reroute($destination ? $destination : '@index');
        }
      // user exists
      } else {
        $this->view->data['form'] = [
          'first_name'  => $firstName,
          'last_name'   => $lastName,
          'email'       => $email,
          'password'    => $password,
        ];
        $this->view->data['errors']['email'] = $f3->get('text.user_exists');
      }
    }
  }


  /**
   * Password reset
   *
   * @param $f3
   * @param $params
   */
  public function password($f3, $params) {
    if ($f3->get('VERB')=='POST') {
      $emailTo = $f3->get('POST.email');
      $user = new User();
      // if such user exists
      if ($user->load('email', $emailTo)->get()) {
        $password = Helper::secureToken(10);
        $f3->set('vars.email', [
          'home_url'    => Helper::baseUrl(),
          'login_url'   => Helper::baseUrl() . substr($f3->alias('login'), 1),
          'password'    => $password,
        ]);
        $email = new Email($emailTo, $f3->get('text.password_recovery'), 'email/password.php');
        if ($email->status()) {
          $user->update(['password' => password_hash($password, PASSWORD_BCRYPT)]);
        } else {
          $this->view->data['error'] = $f3->get('text.email_error');
        }
      }
    }
  }

  private function startSession($user) {
    $f3 = \Base::instance();
    // create a JWT token
    if ($token = JWT::encode([
      'iss' => Helper::baseUrl(),
      'iat' => time(),
      'exp' => time() + $f3->get('SECURITY.LOGIN_PERIOD'),
      'user' => $user
    ],
      $f3->get('SECURITY.TOKEN'),
      $f3->get('SECURITY.ALGORITHM')
    )) {
      setcookie('cc_token', $token, time() + $f3->get('SECURITY.LOGIN_PERIOD'));
    }
  }

  private function endSession() {
    setcookie('cc_token', '', 0);
  }
}