<?php

namespace Controller;

use Inc\Config;
use Inc\Flash;
use Inc\Helper;
use Inc\Logger;
use Inc\Phinx;
use Model\Action;
use Model\Coin;
use Model\CurrencyRate;
use Model\Query;
use Inc\Image;

class Backend extends Base {

  /**
   * Before a page is served
   * @param $f3
   * @param $params
   */
  public function beforeroute($f3, $params) {
    parent::beforeroute($f3, $params);
    $this->view->templateFolder = 'backend';

    $user = $this->user();
    // not logged in
    if (!$user) {
      // capture current URI, so user can be redirected automatically upon logging in
      $f3->reroute('@login?destination=' . urlencode(Helper::uri()));
    // not admin
    } elseif (!$user['admin']) {
      $f3->reroute('@index');
    // admin
    } elseif ($f3->get('ALIAS') != 'backend_update' && Config::instance()->updateRequired()) {
      $f3->reroute('@backend_update');
    }
  }

  /**
   * After page is routed
   * @param $f3
   * @param $params
   */
  public function afterroute($f3, $params) {
    // set title and meta variables
    $alias = $f3->get('ALIAS');
    $this->view->data['page_title'] = $f3->get('site.title')[$alias];
    $this->view->data['page_meta_description'] = $f3->get('site.meta_description')[$alias];

    parent::afterroute($f3, $params);
  }

  /**
   * Backend settings page
   * @param $f3
   * @param $params
   */
  public function settings($f3, $params) {
    $query = new Query();
    $currencyRate = CurrencyRate::instance();
    $this->view->data['languages'] = Helper::languages();
    $this->view->data['colors'] = $this->colors();
    $this->view->data['coins_list'] = $query->coinsList();
    $this->view->data['currencies_list'] = $currencyRate->all();
    $this->view->data['delimiters'] = [ord('.') => '.', ord(',') => ',', ord(' ') => 'space', ord(':') => ':', ord(';') => ';', ord('-') => '-'];

    // if settings are changed and saved
    if ($f3->get('VERB')=='POST') {
      
      if(isset($_POST['site_logo_udf_remove'])&&file_exists('assets/images/logo-udf.png')){
        unlink('assets/images/logo-udf.png');
      }
      if(isset($_FILES['site_logo_udf'])){
        $img = new Image($_FILES['site_logo_udf']['tmp_name']);
        $img->save('assets/images/logo-udf.png');
      }
      
      if (Config::instance()->update($f3->get('POST'))) {
        Flash::instance()->addMessage($f3->get('text.settings_saved'), 'success');
      } else {
        Flash::instance()->addMessage($f3->get('text.settings_not_saved'), 'error');
      }
    }
  }
  public function banners($f3, $params) {
    if(!file_exists('app/templates/public/blocks'))mkdir('app/templates/public/blocks');
    if ($f3->get('VERB')=='POST') {
      if(isset($_POST['banner_before']))file_put_contents('app/templates/public/blocks/before-content.php',$_POST['banner_before']);
      if(isset($_POST['banner_after']))file_put_contents('app/templates/public/blocks/after-content.php',$_POST['banner_after']);
    }
    $this->view->data['banner_before']='';
    $this->view->data['banner_after']='';
    if(file_exists('app/templates/public/blocks/before-content.php'))$this->view->data['banner_before']=file_get_contents('app/templates/public/blocks/before-content.php');
    if(file_exists('app/templates/public/blocks/after-content.php'))$this->view->data['banner_after']=file_get_contents('app/templates/public/blocks/after-content.php');
    
    
  }
  /**
   * Backend view logs page
   * @param $f3
   * @param $params
   */
  public function logs($f3, $params) {
    if ($f3->get('VERB')=='POST') {
      Logger::purge();
    }

    $this->view->data['logs'] = file_exists(Logger::LOG_FILE) ? file_get_contents(Logger::LOG_FILE) : NULL;
  }


  /**
   * Coins management page
   * @param $f3
   * @param $params
   */
  public function coins($f3, $params) {
    if ($f3->get('VERB')=='POST') {
      $action = $f3->get('POST.action');
      if ($action == 'hide_unhide') {
        $symbol = $f3->get('POST.symbol');
        $value = $f3->get('POST.value');

        $coin = new Coin($symbol);
        $coin->update(['active' => $value]);
      }
    }

    $query = new Query();
    $page = $f3->get('GET.page') ? intval($f3->get('GET.page')) : 1;
    $search = $f3->get('GET.search');
    $this->view->data['coins'] = $query->coinsListSearch($page, $search);
    $this->view->data['pages_count'] = ceil($query->countCoins($search)/intval($f3->get('SITE.COINS_LIMIT')));
  }


  /**
   * Coin edit page
   * @param $f3
   * @param $params
   */
  public function coin($f3, $params) {
    $coin = new Coin($params['symbol']);

    if ($f3->get('VERB')=='POST') {
      $coin->update([
        'name'        => $f3->get('POST.name'),
        'website'     => $f3->get('POST.website'),
        'twitter'     => $f3->get('POST.twitter'),
        'proof_type'  => $f3->get('POST.proof_type'),
        'description' => $f3->get('POST.description'),
        'features'    => $f3->get('POST.features'),
        'technology'  => $f3->get('POST.technology'),
        'active'      => $f3->get('POST.active')=='on' ? 1 : 0,
        'featured'    => $f3->get('POST.featured')=='on' ? 1 : 0,
        'ws_disabled' => $f3->get('POST.ws_disabled')=='on' ? 1 : 0,
      ]);
    }

    $this->view->data['coin'] = $coin->get(['symbol', 'name', 'proof_type', 'website', 'twitter', 'description', 'features', 'technology', 'supply', 'total_supply', 'active', 'featured','ws_disabled']);

    if (empty($this->view->data['coin']))
      $f3->reroute('@backend_coins');
  }

  /**
   * Backend cron page
   * @param $f3
   * @param $params
   */
  public function cron($f3, $params) {
    if ($f3->get('VERB')=='POST') {
      set_time_limit(1800); // set time limit to 30 minutes
      $action = new Action();
      // update FX rates
      if ($updatedCount = $action->updateCurrencyRates()) {
        Flash::instance()->addMessage($f3->get('text.currencies_quotes_updated', $updatedCount), 'success');
      } else {
        Flash::instance()->addMessage($f3->get('text.currencies_quotes_not_updated'), 'error');
      }

      // update coins data
      if ($updatedCount = $action->updateCoinData()) {
        Flash::instance()->addMessage($f3->get('text.coins_quotes_updated', $updatedCount), 'success');
      } else {
        Flash::instance()->addMessage($f3->get('text.coins_quotes_not_updated'), 'error');
      }
      
      $action->updateSitemap();
    }
  }

  /**
   * Update config & database (migrations)
   * @param $f3
   * @param $params
   */
  public function update($f3, $params) {
    if ($f3->get('VERB')=='POST') {
      $phinx = new Phinx();
      $phinx->migrate();
      if (Config::instance()->addMissingVariablesFromMasterConfig()) {
        Flash::instance()->addMessage($f3->get('text.update_config_success'), 'success');
      } else {
        Flash::instance()->addMessage($f3->get('text.update_config_error'), 'error');
      }
    }

    $this->view->data['update_required'] = Config::instance()->updateRequired();
  }

  /**
   * Backend help page
   * @param $f3
   * @param $params
   */
  public function help($f3, $params) {
    // nothing here yet
  }

  /**
   * Backend language strings overrides page
   * @param $f3
   * @param $params
   */
  public function languageStrings($f3, $params) {
    $lang = $f3->get('GET.language') ? $f3->get('GET.language') : $f3->get('LANG');
    $this->view->data['languages'] = Helper::languages();
    $this->view->data['strings'] = include('app/dict/'.$lang.'.php');

    if ($f3->get('VERB')=='POST') {
      $overrides = [];
      foreach ($f3->get('POST') as $code => $override) {
        $code = str_replace('#','.',$code);
        if (is_string($override)) {
          $overrides[$code] = $override != '' ? $override : $this->view->data['strings'][$code];
        } elseif (is_array($override)) {
          foreach ($override as $sCode => $sOverride) {
            $overrides[$code][$sCode] = $sOverride != '' ? $sOverride : $this->view->data['strings'][$code][$sCode];
          }
        }
      }
      if (!empty($overrides)) {
        file_put_contents('app/dict-udf/'.$lang.'.php', '<?php return '.var_export($overrides, TRUE).';');
      }
    }

    $this->view->data['overrides'] = file_exists('app/dict-udf/'.$lang.'.php') ? include('app/dict-udf/'.$lang.'.php') : [];
  }

  /**
   * Download new coins and add them to the local database
   * @param $f3
   * @param $params
   */
  public function downloadCoins($f3, $params) {
    $this->view->data['finfo_enabled'] = Helper::finfoEnabled();
    $this->view->data['gd_enabled'] = Helper::gdEnabled();
    $this->view->data['imagick_enabled'] = Helper::imagickEnabled();

    if ($f3->get('VERB')=='POST') {
      set_time_limit(1800);
      $updateSubs = $f3->get('POST.update_subs')=='on' ? TRUE : FALSE;
      // calculate number of existing coins
      $query = new Query();
      $coinsCount = $query->countCoins();
      // download new coins data
      $action = new Action();
      $downloadedCoinsCount = $action->downloadCoins($updateSubs);
      // run coin seeder
      $phinx = new Phinx();
      $phinx->seedCoin();
      // calculate number of inserted coins
      $insertedCoins = $query->countCoins() - $coinsCount;
      // display flash messages
      Flash::instance()->addMessage($f3->get('text.coins_downloaded', [$downloadedCoinsCount]), 'success');
      Flash::instance()->addMessage($f3->get('text.coins_inserted', [$insertedCoins]), 'success');
    }
  }

  private function colors() {
    $result = [];
    $f3 = \Base::instance();
    foreach (glob('assets/vendor/materialize-css/css/*.css') as $file) {
      if (preg_match('#([a-z]+)\.css$#', $file, $matches) && $matches[1]!= 'widget')
        $result[] = $matches[1];
    }
    return $result;
  }
}