<?php

namespace Inc;

/**
 * Class Config - Singleton
 * @package Helper
 */
class Config extends \Prefab {
  private $localConfigFileName  = 'config/framework.ini';
  private $masterConfigFileName = 'config/.framework.ini';
  private $f3;
  private $config;

  // using protected constructor for singleton, so it can't be instantiated like a normal class
  protected function __construct() {
    $this->f3 = \Base::instance();
    if (file_exists($this->localConfigFileName)) {
      $this->config = parse_ini_file($this->localConfigFileName, TRUE, INI_SCANNER_RAW);
    } else {
      Logger::log('Config file does not exist!');
    }
  }

  /**
   * Update config variables
   * @param $configVariables
   */
  public function update($configVariables) {
    foreach ($configVariables as $configPath => $configValue) {
      if (strpos($configPath, '#') !== FALSE) {
        list ($configGroup, $configName) = explode('#', $configPath);
        // if such config parameter exists
        if (isset($this->config[$configGroup][$configName])) {
          // if array is passed turn it in comma delimited string
          if (is_array($configValue)) {
            $configValue = implode(',', $configValue);
          }

          // update config parameter
          $this->config[$configGroup][$configName] = $configValue;

          // it's important to transform a string to an array if there are comma-delimited values, so $f3->set() works
          $configArrayValue = explode(',', $configValue);
          if (count($configArrayValue) > 1 && !in_array($configGroup, ['DB','SMTP'])) {
            $configValue = $configArrayValue;
          }

          // set current value via f3, so changes take immediate effect
          // globals prefix should be removed, so for example it equals this:
          // $this->f3->set('LANGUAGE', 'es-ES');
          $this->f3->set(str_replace(['globals#','#'],['','.'],$configPath), $configValue);
          if ($configPath == 'globals#LANGUAGE')
            $this->f3->set('LANG', $configValue);
        }
      }
    }
    return $this->save();
  }

  /**
   * add new (missing) config variables from the master config file
   * @return int
   */
  public function addMissingVariablesFromMasterConfig() {
    if (!empty($this->config) && ($masterConfig = parse_ini_file($this->masterConfigFileName, TRUE, INI_SCANNER_RAW))) {
      foreach ($masterConfig as $masterConfigGroup => $masterConfigGroupValues) {
        foreach ($masterConfigGroupValues as $configName => $configValue) {
          if (!isset($this->config[$masterConfigGroup][$configName]))
            $this->config[$masterConfigGroup][$configName] = $configValue;
        }
      }
    }
    return $this->save();
  }

  /**
   * Check if the master config file was recently updated (usually during new release installation),
   * in which case new config variables need to be added to the local config file.
   *
   * @return bool
   */
  public function updateRequired() {
    return file_exists($this->localConfigFileName) && file_exists($this->masterConfigFileName) && filemtime($this->masterConfigFileName) > filemtime($this->localConfigFileName);
  }


  /**
   * Save config file
   * @return int - number of bytes written
   */
  private function save() {
    $strings = [];
    foreach ($this->config as $configGroup => $configGroupValues) {
      $strings[] = '['.$configGroup.']';
      foreach ($configGroupValues as $configName => $configValue) {
        // add double quotes to values where special symbols can be used. It's also to important to quote strings where ";" is present,
        // otherwise parse_ini_file() will ignore the rest of the string after semi-colon
        if (in_array($configGroup.'.'.$configName, ['globals.LOCALES','SITE.RSS_URL','SITE.COIN_BUY_LINK','SITE.COIN_SELL_LINK','DB.PASSWORD'])) {
          $configValue = '"' . $configValue . '"';
        }
        $strings[] = $configName . '=' . $configValue;
      }
      $strings[] = '';
    }

    return !empty($this->config) ? file_put_contents($this->localConfigFileName, implode("\n", $strings)) : NULL;
  }
}