<?php
namespace Model;
use DB\SQL\Mapper;
use Inc\Helper;
use Inc\HTTP;
use Inc\Logger;

/**
 * Class Coin - functions for a specific coin
 * @package Model
 */
class Coin extends Base {
	const COINS_HISTORICAL_MARKET_DATA_URL = 'https://min-api.cryptocompare.com/data/%s?fsym=%s&tsym=%s&limit=%d&aggregate=%d&api_key=%s';
	const HISTORY_DATA_FOLDER = 'app/data/history';
	const MAX_INTRADAY_HISTORY_LENGTH = 288; // 24 hours of 5 min intervals (60/5 * 24)
	private $cacheTime = array('histoday' => 7200, 'histohour' => 3600, 'histominute' => 900);
	private $symbol;

	public function __construct($symbol) {
		parent::__construct();
		$this->symbol = $symbol;

		return $this;
	}

	/**
	 * Get historical data
	 *
	 * @param string $interval - histoday, histohour, histominute
	 * @param int $maxDataPoints
	 * @param int $discardDataEarlierThan - don't save quotes timestamped earlier than the given date
	 * @param int $aggregatePoints
	 * @return array|mixed
	 */
	public function history($interval = 'histoday', $maxDataPoints = 2000, $discardDataEarlierThan = NULL, $aggregatePoints = 1) {
		$result = [];
		$currency = 'USD';
		$historyDataFileName = self::HISTORY_DATA_FOLDER . '/' . str_replace('*','@',$this->symbol) . '_' . $currency . '_' . $interval . '_' . $aggregatePoints . '.json';
		if (!file_exists($historyDataFileName) || (time()-filemtime($historyDataFileName)) > $this->cacheTime[$interval]) {
			$requestUrl = sprintf(self::COINS_HISTORICAL_MARKET_DATA_URL, $interval, $this->symbol, $currency, $maxDataPoints, $aggregatePoints, $this->f3->get('API.CRYPTOCOMPARE_API_KEY'));
			Logger::log($requestUrl, 3);
			if ($response = HTTP::get($requestUrl)) {
				$json = json_decode($response);

				if (isset($json->Response) && $json->Response == 'Success' && isset($json->Data)) {
					Logger::log(sprintf('Retrieved rows: %d', count($json->Data)));
					foreach ($json->Data as $i => $item) {
						if (!$discardDataEarlierThan || $item->time > $discardDataEarlierThan) {
							$result[] = [
								'date' => $item->time,
								'open' => floatval($item->open),
								'high' => floatval($item->high),
								'low' => floatval($item->low),
								'value' => floatval($item->close),
								'volume' => floatval($item->volumeto),
							];
						}
					}
					// save daily history to avoid unnecessary API calls
					if (!empty($result))
						file_put_contents($historyDataFileName, json_encode($result));
				}
			}
		} else {
			Logger::log(sprintf('Reading history data for %s (%s,%d,%d)', $this->symbol, $interval, $maxDataPoints, $aggregatePoints), 3);
			$result = json_decode(file_get_contents($historyDataFileName), JSON_OBJECT_AS_ARRAY);
		}

		// format output
		$displayCurrencyRate = CurrencyRate::instance()->rate();
		array_walk($result, function(&$item) use ($displayCurrencyRate, $interval) {
			$item['date_fmt'] = Helper::timeStamp('@'.$item['date'], NULL, NULL, NULL, 'Y-m-d');
			$item['date'] *= 1000;
			$item['open'] *= $displayCurrencyRate;
			$item['high'] *= $displayCurrencyRate;
			$item['low'] *= $displayCurrencyRate;
			$item['value'] *= $displayCurrencyRate;
			$item['volume'] *= $displayCurrencyRate;
			$item['interval'] = $interval;
		});

		return $result;
	}

	/**
	 * Get combined history (daily & intraday)
	 */
	function dailyIntradayHistory() {
		$dailyData = $this->history();
		$latestCloseDate = $dailyData[count($dailyData)-1]['date']/1000;
		$intradayData = $this->history('histominute', intval(date('H'))*(60/15), $latestCloseDate, 15);
		$history = array_merge($dailyData, $intradayData);
		return $history;
	}

	/**
	 * Get coin data from DB
	 * @return mixed
	 */
	public function data() {
		$displayCurrencySymbol = CurrencyRate::instance()->symbol();
		$displayCurrencyRate = CurrencyRate::instance()->rate();
		$result = $this->db->exec('SELECT symbol, name, COALESCE(logo, "_NOLOGO.png") AS logo, price, `change`, change_pct,
																			open, low, high, supply, market_cap, volume, volume_ccy, total_volume, total_volume_ccy, last_updated,
																			proof_type, website, twitter, description, features, ws_disabled, technology, total_supply, subs,
																			market_cap / (SELECT SUM(market_cap)/100 FROM coin WHERE active = 1) AS market_share,
																			price / (SELECT price FROM coin WHERE symbol = "BTC") AS btc_price
																		FROM coin
																		WHERE symbol = :symbol AND active = 1', [':symbol' => $this->symbol])[0];

		if (!empty($result)) {
			array_walk($result, function(&$value, $key) use ($displayCurrencySymbol, $displayCurrencyRate) {
				if (in_array($key, ['price','change','open','high','low','market_cap','volume_ccy','total_volume_ccy'])) {
					$value *= $displayCurrencyRate;
				} elseif ($key == 'last_updated') {
					$value = Helper::timeStamp($value, NULL, NULL, NULL, 'Y-m-d H:i:s P T');
				} elseif ($key == 'subs') {
					// filter subscriptions to leave only those with current coin and currency
					$subs = array_filter(explode(',', $value), function($sub) use($displayCurrencySymbol) {
						return preg_match('#~' . $this->symbol . '~' . $displayCurrencySymbol . '$#', $sub);
					});
					// prepend subscription type, i.e. 0~
					array_walk($subs, function(&$sub) {
						$sub = '0~' . $sub;
					});
					$value = implode(',', $subs);
				}
			});
		}

		return $result;
	}

	/**
	 * Get coin info
	 *
	 * @param array $fields - what fields to retrieve
	 * @return array
	 */
	public function get(array $fields = ['symbol'], $currencySymbol = NULL) {
		$coin = [];
		$currency = $currencySymbol ? (new Currency($currencySymbol))->get(['symbol','name','rate']) : CurrencyRate::instance()->get();
		$fields = array_map(function($field) {
			if ($field=='logo')
				return 'COALESCE(logo, "_NOLOGO.png") AS logo';
			elseif ($field=='change')
				return '`change`';
			else
				return $field;
		}, $fields);

		$result = $this->db->exec('SELECT ' . implode(',', $fields) . ' FROM coin WHERE symbol = :symbol', [':symbol' => $this->symbol]);

		if (isset($result[0]) && !empty($result[0])) {
			$coin = $result[0];
			// convert price to given currency
			array_walk($coin, function(&$value, $key) use ($currency) {
				if (in_array($key, ['price','change','open','high','low','market_cap','volume_ccy','total_volume_ccy'])) {
					$value *= $currency['rate'];
				}
			});
			// add extra fields
			$coin['currency_symbol'] = $currency['symbol'];
			$coin['currency_rate'] = $currency['rate'];
		}

		return $coin;
	}

	/**
	 * Update coin market data
	 * @param $fields
	 * @return mixed
	 */
	public function update($fields) {
		$coinTable = new Mapper($this->db->connection(), 'coin');
		$coinTable->load(['symbol = :symbol', ':symbol' => $this->symbol]);
		if (!$coinTable->dry()) {
			if (isset($fields['price'])) {
				$fields['price'] = min($fields['price'], 1E+10-.01); // prevent Numeric value out of range MySQL error
				// append latest price to intraday quotes
				$intradayQuotes = $coinTable->intraday_quotes ? json_decode($coinTable->intraday_quotes, JSON_OBJECT_AS_ARRAY) : [];
				if (isset($intradayQuotes['t']) && isset($intradayQuotes['p'])) {
					$count = count($intradayQuotes['t']);
					// check if number of quotes exceeds limit, in which case remove the oldest elements
					if ($count >= self::MAX_INTRADAY_HISTORY_LENGTH) {
						$intradayQuotes['t'] = array_slice($intradayQuotes['t'], -self::MAX_INTRADAY_HISTORY_LENGTH+1);
						$intradayQuotes['p'] = array_slice($intradayQuotes['p'], -self::MAX_INTRADAY_HISTORY_LENGTH+1);
					}
				}
				$fields['intraday_quotes'] = json_encode(array_merge_recursive($intradayQuotes, ['t' => [time()], 'p' => [$fields['price']]]));
			}

			// prevent Numeric value out of range MySQL error
			if (isset($fields['open']))         $fields['open']         = min($fields['open'],          1E+10-.01);
			if (isset($fields['low']))          $fields['low']          = min($fields['low'],           1E+10-.01);
			if (isset($fields['high']))         $fields['high']         = min($fields['high'],          1E+10-.01);
			if (isset($fields['supply']))       $fields['supply']       = min($fields['supply'],        1E+28-.99);
			if (isset($fields['market_cap']))   $fields['market_cap']   = min($fields['market_cap'],    1E+28-.99);
			if (isset($fields['volume']))       $fields['volume']       = min($fields['volume'],        1E+28-.99);
			if (isset($fields['total_volume'])) $fields['total_volume'] = min($fields['total_volume'],  1E+28-.99);
			if (isset($fields['volume_ccy']))   $fields['volume_ccy']   = min($fields['volume_ccy'],    1E+28-.99);
			if (isset($fields['total_volume_ccy'])) $fields['total_volume_ccy'] = min($fields['total_volume_ccy'], 1E+28-.99);
			if (isset($fields['total_supply'])) $fields['total_supply'] = min($fields['total_supply'],  1E+28-.99);
			// take into account that change can be negative
			if (isset($fields['change']))       $fields['change']       = Helper::sign($fields['change']) * min(abs($fields['change']), 1E+10-.01);
			if (isset($fields['change_pct']))   $fields['change_pct']   = Helper::sign($fields['change_pct']) * min(abs($fields['change_pct']), 1E+8-.01);

			$coinTable->copyfrom($fields);
		}
		return $coinTable->save();
	}

	/**
	 * Is the coin in the favorites list for current user
	 * @return bool
	 */
	public function isFavorite() {
		$user = $this->f3->get('USER');
		if ($user['id']) {
			$count = $this->db->exec(
				'SELECT COUNT(*) AS cnt FROM favorite_coin WHERE user_id = :user_id AND coin_id = (SELECT id FROM coin WHERE symbol = :symbol)',
				[':user_id' => $user['id'], ':symbol' => $this->symbol]
			)[0]['cnt'];
			return $count ? TRUE : FALSE;
		}

		return FALSE;
	}

	/**
	 * Add the coin to favorites list of current user
	 * @return bool
	 */
	public function addFavorite() {
		$user = $this->f3->get('USER');
		if ($user['id']) {
			$coinTableMapper = new Mapper($this->db->connection(), 'coin');
			$coinTableMapper->load(['symbol = :symbol', ':symbol' => $this->symbol]);
			if (!$coinTableMapper->dry()) {
				Logger::log(sprintf('Adding %s to favorites list of user %d', $this->symbol, $user['id']));
				$favoriteCoinTableMapper = new Mapper($this->db->connection(), 'favorite_coin');
				$favoriteCoinTableMapper->copyfrom(['user_id' => $user['id'], 'coin_id' => $coinTableMapper->id, 'created' => Helper::timeStamp()]);
				try {
					$favoriteCoinTableMapper->insert();
				} catch (\Exception $e) {
					Logger::log($e->getMessage());
				}
				return $favoriteCoinTableMapper->get('_id') ? TRUE : FALSE;
			}
		}

		return FALSE;
	}

	/**
	 * Remove the coin from favorites list of current user
	 * @return bool
	 */
	public function removeFavorite() {
		$user = $this->f3->get('USER');
		if ($user['id']) {
			$coinTableMapper = new Mapper($this->db->connection(), 'coin');
			$coinTableMapper->load(['symbol = :symbol', ':symbol' => $this->symbol]);
			if (!$coinTableMapper->dry()) {
				Logger::log(sprintf('Removing %s from favorites list of user %d', $this->symbol, $user['id']));
				$favoriteCoinTableMapper = new Mapper($this->db->connection(), 'favorite_coin');
				$favoriteCoinTableMapper->load(['user_id = :user_id AND coin_id = :coin_id', ':user_id' => $user['id'], ':coin_id' => $coinTableMapper->id]);
				if (!$favoriteCoinTableMapper->dry()) {
					return $favoriteCoinTableMapper->erase() ? TRUE : FALSE;
				}
			}
		}

		return FALSE;
	}
}