"use strict";
$(document).ready(function () {
  var historicalDataTableInitialized = false;
  var utcHours = new Date().getUTCHours();
  var utcMinutes = new Date().getUTCMinutes();
  var PERIODS = {
    '1d': {period: 'mm', count: utcHours>0 ? (utcHours-1)*60+utcMinutes : 60, label: TEXT['1d']},
    '1w': {period: 'DD', count: 7, label: TEXT['1w']},
    '2w': {period: 'DD', count: 14, label: TEXT['2w']},
    '1m': {period: 'MM', count: 1, label: TEXT['1m']},
    '3m': {period: 'MM', count: 3, label: TEXT['3m']},
    '6m': {period: 'MM', count: 6, label: TEXT['6m']},
    'ytd':{period: 'YTD', label: TEXT['ytd']},
    '1y': {period: 'YYYY', count: 1, label: TEXT['1y']},
    '2y': {period: 'YYYY', count: 2, label: TEXT['2y']},
    '5y': {period: 'YYYY', count: 5, label: TEXT['5y']},
    'all':{period: 'MAX', label: TEXT.all}
  };
  var screenWidth = (window.innerWidth > 0) ? window.innerWidth : screen.width;
  var $chartContainer = $('#coin-chart-container');
  var $chartPreloader = $('#coin-chart-preloader');
  var $comparisonSelect = $('#select-comparison');
  var $tradesTableBody = $('#trades table tbody');
  var $coinPrice = $('#coin-price-value');
  var $coinAbsChange = $('#coin-abs-change-value');
  var $coinPctChange = $('#coin-pct-change-value');
  var $addFavoriteButton = $('#add-favorite-button');
  var $removeFavoriteButton = $('#remove-favorite-button');
  var coinLastPrice = numeral(COIN_PRICE).value();
  var displayedTradesCount = 0;
  var chart;
  var loadedComparisonSeries = [];
  var comparisonSelectSymbols = [];
  var primaryColor = COLOR_CODE;
  var secondaryColor = '#000';
  
  $('.tooltip').tooltip();

  // translations for amCharts export dropdown
  AmCharts.translations['export'][LANGSHORT] = TEXT.amCharts;

  AmCharts.translations.dataLoader[LANGSHORT] = {
    'Error loading the file': 'Error loading the file',
    'Error parsing JSON file': 'Error parsing JSON file',
    'Unsupported data format': 'Unsupported data format',
    'Loading data...': '' // don't display "Loading data..."
  };

  moment.locale(LANGSHORT);

  var buildChart = function() {

    var chartOptions = {
      type: 'stock',
      language: LANGSHORT,

      categoryAxesSettings: {
        minPeriod: 'mm',
        color: secondaryColor,
        gridColor: primaryColor, // vertical grid color
        gridAlpha: 0.1,
        gridThickness: 1,
        groupToPeriods: ['15mm','30mm','DD','WW','MM'],
        equalSpacing: true // skip time gaps
      },

      dataSets: [{
        dataLoader: {
          url: API.history + '/' + COIN_SYMBOL,
          format: 'json',
          postProcess: function(data, config, chart) {
            if(!historicalDataTableInitialized) {
              displayHistoricalDataTable(data.history);
              historicalDataTableInitialized = true;
            }
            return data.history;
          }
        },
        title: COIN_SYMBOL,
        fieldMappings: [{
          fromField: 'value',
          toField: 'value'
          }, {
            fromField: 'volume',
            toField: 'volume'
          }
        ],
        categoryField: 'date'
      }],

      panelsSettings: {
        usePrefixes: CHART.usePrefixes,
        creditsPosition: 'bottom-left'
      },

      panels: [{
        showCategoryAxis: true,
        title: TEXT.price,
        percentHeight: 70,
        precision: 2,
        drawingIconsEnabled: true,
        eraseAll: true,
        stockGraphs: [ {
          id: 'g1',
          type: CHART.typeCoin,
          valueField: 'value',
          lineColor: primaryColor,
          fillAlphas: 0.15,
          lineThickness: 2,
          comparable: true,
          compareGraph: {
            type:           CHART.typeCoin == 'candlestick' ? 'smoothedLine' : CHART.typeCoin,
            fillAlphas:     0.15,
            lineThickness:  2
          },
          balloonFunction: priceChartBalloonText,
          compareGraphBalloonFunction: priceChartBalloonText,
          useDataSetColors: false
        }],
        stockLegend: {
          color: secondaryColor,
          valueText:                DISPLAY_CURRENCY.code + '[[value]]',
          valueTextRegular:         DISPLAY_CURRENCY.code + '[[value]]',
          valueTextComparing:       '[[percents.value]]%',
          periodValueText:          DISPLAY_CURRENCY.code + '[[value.close]]',
          periodValueTextRegular:   DISPLAY_CURRENCY.code + '[[value.close]]',
          periodValueTextComparing: '[[percents.value.close]]%'
        },
        valueAxes: [{
          position: 'right',
          color: secondaryColor, // color of values
          gridColor: primaryColor, //horizontal grid line color
          gridAlpha: 0.1,
          gridThickness: 1
        }],
        numberFormatter: {
          precision: -1,
          decimalSeparator: DECIMAL_POINT,
          thousandsSeparator: THOUSANDS_SEPARATOR
        }
      }, {
        title: TEXT.volume,
        percentHeight: 30,
        precision: 0,
        stockGraphs: [{
          valueField: 'volume',
          type: CHART.typeVolume,
          showBalloon: true,
          lineColor: primaryColor,
          lineThickness: 1,
          fillAlphas: 0.3,
          periodValue: 'Sum',
          comparable: true,
          compareGraph:     {
            type:             CHART.typeVolume,
            fillAlphas:       0.3,
            lineThickness:    1,
            balloonText:      '[[title]]: <b>' + DISPLAY_CURRENCY.code + '[[value]]</b>'
          },
          balloonText:      '[[title]]: <b>' + DISPLAY_CURRENCY.code + '[[value]]</b>',
          useDataSetColors: false
        }],
        stockLegend: {
          valueText:                DISPLAY_CURRENCY.code + '[[value]]',
          valueTextRegular:         DISPLAY_CURRENCY.code + '[[value]]',
          valueTextComparing:       '[[percents.value]]%',
          periodValueText:          DISPLAY_CURRENCY.code + '[[value.close]]',
          periodValueTextRegular:   DISPLAY_CURRENCY.code + '[[value.close]]',
          periodValueTextComparing: '[[percents.value.close]]%'
        },
        valueAxes: [{
          position: 'right',
          gridColor: primaryColor, //horizontal grid line color
          gridAlpha: 0.1,
          gridThickness: 1
        }],
        numberFormatter: {
          precision: -1,
          decimalSeparator: DECIMAL_POINT,
          thousandsSeparator: THOUSANDS_SEPARATOR
        }
      }],

      chartScrollbarSettings: {
        graph: 'g1',
        enabled: CHART.scrollbarEnabled,
        graphFillColor: primaryColor,
        graphFillAlpha: 0.4,
        selectedGraphFillColor: primaryColor,
        selectedGraphFillAlpha: 1
      },

      chartCursorSettings: {
        valueBalloonsEnabled: true,
        graphBulletSize: 1,
        valueLineBalloonEnabled: true,
        valueLineEnabled: true,
        valueLineAlpha: 1,
        categoryBalloonColor: primaryColor,
        categoryBalloonAlpha: 0.8,
        cursorColor: primaryColor,
        cursorAlpha: 0.8
      },

      periodSelector: {
        position: 'top',
        periodsText: '',
        inputFieldsEnabled: false, //disable dates input
        periods: [] // fill it later
      },

      dataSetSelector: {
        position: '' // leave empty to hide the native dataSet selection control
      },

      comparedDataSets: [],

      export: {
        enabled: true
      },

      listeners: [{
        event: 'init',
        method: function () {
          chartRemoveLoadingState();
          // add chart background image when the chart is initialized
          $chartContainer.find('.amcharts-stock-panel-div-stockPanel0 .amcharts-main-div')
            .prepend('<div style="background: url(assets/images/coins/'+COIN_LOGO+') no-repeat center center; position: absolute; width: 100%; height: 100%; opacity: 0.18;"></div>');
        }
      }]
    };
    var initChart = function () {
        
      // if(get_cookie('black_theme')){
      if( $('body').hasClass('black-theme') ){
        chartOptions.theme="dark";
        chartOptions.categoryAxesSettings.color = '#aaa';
        chartOptions.panels[0].stockLegend.color = '#aaa';
        chartOptions.panels[0].valueAxes[0].color = '#aaa';
      }
      else{
        delete chartOptions.theme;
        chartOptions.categoryAxesSettings.color = secondaryColor;
        chartOptions.panels[0].stockLegend.color = secondaryColor;
        chartOptions.panels[0].valueAxes[0].color = secondaryColor;
      }
      // "theme": "light",
      if(chart)chart.clear();
      chart = AmCharts.makeChart($chartContainer.attr('id'), chartOptions);
    };
    
    // additional settings for candlestick chart
    if (CHART.typeCoin == 'candlestick') {
      chartOptions.panels[0].stockGraphs[0]['openField']        = 'open';
      chartOptions.panels[0].stockGraphs[0]['lowField']         = 'low';
      chartOptions.panels[0].stockGraphs[0]['highField']        = 'high';
      chartOptions.panels[0].stockGraphs[0]['closeField']       = 'value';
      chartOptions.panels[0].stockGraphs[0]['proCandlesticks']  = true;
      chartOptions.dataSets[0].fieldMappings.push({fromField: 'open', toField: 'open'});
      chartOptions.dataSets[0].fieldMappings.push({fromField: 'low', toField: 'low'});
      chartOptions.dataSets[0].fieldMappings.push({fromField: 'high', toField: 'high'});
    }

    // add predefined chart periods
    for (var i=0; i<CHART.periods.length; i++) {
      var periodName = CHART.periods[i];
      if (typeof PERIODS[periodName] != 'undefined') {
        //chartOptions.periodSelector.periods.push(PERIODS[periodName]);
        chartOptions.periodSelector.periods[i] = PERIODS[periodName];
        // set default (selected) chart period
        if (periodName == CHART.defaultPeriod) {
          chartOptions.periodSelector.periods[i].selected = true;
        }
      }
    }

    
    initChart();

    /**
     * Format balloon for coin price chart
     * @param item
     * @param graph
     * @returns {*}
     */
    function priceChartBalloonText(item, graph) {
      var result;
      if (graph.type == 'candlestick') {
        var open  = DISPLAY_CURRENCY.code + formatDecimal(item.values.open);
        var high  = DISPLAY_CURRENCY.code + formatDecimal(item.values.high);
        var low   = DISPLAY_CURRENCY.code + formatDecimal(item.values.low);
        var close = DISPLAY_CURRENCY.code + formatDecimal(item.values.close);
        result = graph.title + '<hr><table class="charts-tooltip"><tbody><tr><td class="right-align">'+TEXT.open+':</td><td><b>'+open+'</b></td></tr><tr><td class="right-align">'+TEXT.high+':</td><td><b>'+high+'</b></td></tr><tr><td class="right-align">'+TEXT.low+':</td><td><b>'+low+'</b></td></tr><tr><td class="right-align">'+TEXT.close+':</td><td><b>'+close+'</b></td></tr></tbody></table>';
      } else {
        var value = formatDecimal(item.values.value);
        result = graph.title + '<hr>' + TEXT.close + ': <b>' + DISPLAY_CURRENCY.code + value + '</b>';
      }

      return result;
    }

    return chart;
  };
  
  
  var initRealTimeFeed = function() {
    // add aggregate quotes to subscription, always subscribe to USD quotes and then convert them to current display currency
    // subscribe to current coin only if real-time feed is enabled
    if (COINS_REALTIME_FEED&&!COIN_WS_DISABLED)
      COIN_TRADE_SUBS.push('5~CCCAGG~'+COIN_SYMBOL+'~USD');

    if (COIN_TRADE_SUBS.length&&!COIN_WS_DISABLED) {
      var feed = new RealTimeDataFeed({
        subs: COIN_TRADE_SUBS,
        onTrade: function (trade) {
          var $row = $(
            '<tr>' +
            '<td data-title="' + TEXT.id + '"><div>' + trade.ID + '</div><div class="time-ago">'+moment(trade.TS*1000).fromNow()+'</div></td>' +
            '<td data-title="' + TEXT.market + '">' + trade.M + '</td>' +
            '<td data-title="' + TEXT.type + '"><span class="' + (trade.TYPE == 'buy' ? 'label_green' : (trade.TYPE == 'sell' ? 'label_red' : '')) + '"><i class="material-icons">' + (trade.TYPE == 'buy' ? 'arrow_upward' : (trade.TYPE == 'sell' ? 'arrow_downward' : '')) + '</i>' + TEXT['trade_' + trade.TYPE] + '</span></td>' +
            '<td data-title="' + TEXT.price + '" class="right-align"><sup>' + DISPLAY_CURRENCY.code + '</sup>' + formatDecimal(trade.P) + '</td>' +
            '<td data-title="' + TEXT.quantity + '" class="right-align">' + formatDecimal(trade.Q) + '</td>' +
            '<td data-title="' + TEXT.total + '" class="right-align"><sup>' + DISPLAY_CURRENCY.code + '</sup>' + formatDecimal(trade.TOTAL) + '</td>' +
            '</tr>'
          ).prependTo($tradesTableBody).addClass(trade.TYPE == 'buy' ? 'animate-green' : (trade.TYPE == 'sell' ? 'animate-red' : ''));
          setTimeout(function () {
            $row.removeClass('animate-green animate-red');
          }, 500);
          displayedTradesCount++;
          if (displayedTradesCount > TRADES_LIMIT)
            $tradesTableBody.find('tr:last-child').remove();
        },
        onAggQuote: function (quote) {
          if (typeof quote.PRICE != 'undefined') {
            var currentPrice = quote.PRICE * DISPLAY_CURRENCY.rate;
            var lastTickChange = currentPrice - coinLastPrice;
            $coinPrice.text(formatDecimal(currentPrice)).addClass(lastTickChange > 0 ? 'animate-green' : (lastTickChange < 0 ? 'animate-red' : ''));
            setTimeout(function () {
              $coinPrice.removeClass('animate-green animate-red');
            }, 500);
            coinLastPrice = currentPrice;
          }
          if (typeof quote.OPEN24HOUR != 'undefined' && coinLastPrice > 0) {
            var openPrice = quote.OPEN24HOUR * DISPLAY_CURRENCY.rate;
            var absChange = coinLastPrice - openPrice;
            var pctChange = coinLastPrice / openPrice - 1;
            $coinAbsChange.text(formatDecimal(absChange));
            $coinPctChange.text('('+formatPercentage(pctChange)+')');
            if (absChange > 0 && $coinAbsChange.hasClass('change-down'))
              $coinAbsChange.removeClass('change-down').addClass('change-up');
            if (absChange < 0 && $coinAbsChange.hasClass('change-up'))
              $coinAbsChange.removeClass('change-up').addClass('change-down');
            if (pctChange > 0 && $coinPctChange.hasClass('change-down'))
              $coinPctChange.removeClass('change-down').addClass('change-up');
            if (pctChange < 0 && $coinPctChange.hasClass('change-up'))
              $coinPctChange.removeClass('change-up').addClass('change-down');
          }
        }
      });
    } else {
      $tradesTableBody.append('<tr><td colspan="6" class="center-align">'+TEXT.no_subs+'</td></tr>')
    }
  };

  function displayHistoricalDataTable(history) {
    var $table = $('#historical_data table');
    var $tbody = $table.find('tbody');
    var n = history.length;
    for (var i=0; i<n; i++) {
      if (history[i].value > 0 && history[i].interval == 'histoday') {
        $tbody.append('<tr>' +
            '<td>' + history[i].date_fmt + '</td>' +
            '<td class="right-align"><sup>' + DISPLAY_CURRENCY.code + '</sup>' + formatDecimal(history[i].value) + '</td>' +
            '<td class="right-align" data-order="' + history[i].volume + '">' + formatDecimal(history[i].volume) + ' ' + COIN_SYMBOL + '</td>' +
            '</tr>');
      }
    }
    $table.initDataTable(15, [[0, 'desc']]);
  }
  
  function chartSetLoadingState() {
    $chartPreloader.show();
  }
  

  function chartRemoveLoadingState() {
    $chartPreloader.hide();
  }

  chart = buildChart();
  
  initRealTimeFeed();
  
  function addComparison(symbol) {
    log('addComparison', symbol);
    if (typeof chart != 'undefined') {
      chartSetLoadingState();
      // if symbol is not added to comparison already (in which case the data would be already loaded)
      if ($.inArray(symbol, loadedComparisonSeries) == -1) {
        loadedComparisonSeries.push(symbol);

        // load historical data for given symbol
        $.getJSON(API.history + '/' + symbol, function (response) {
          log('historicalData', response);
          if (typeof response.history != 'undefined') {
            var dataSet = {
              title: symbol,
              compared: true,
              fieldMappings: [{
                fromField: 'value',
                toField: 'value'
              }, {
                fromField: 'volume',
                toField: 'volume'
              }
              ],
              dataProvider: response.history,
              categoryField: 'date'
            };
            chart.dataSets.push(dataSet);
            chart.comparedDataSets.push(dataSet);
            chart.validateData();
            chartRemoveLoadingState();
          } else {
            deleteFromArray(loadedComparisonSeries, symbol);
            setTimeout(function () {
              chartRemoveLoadingState();
            }, 3000);
          }
        });
        // If data was already loaded before just add it to comparison
      } else {
        for (var i = 0; i < chart.dataSets.length; i++) {
          if (chart.dataSets[i].title == symbol) {
            chart.dataSets[i].compared = true;
          }
        }
        chart.validateData();
        chartRemoveLoadingState();
      }
    }
  }

  /**
   * Delete coin comparison
   */
  function deleteComparison(symbol) {
    log('deleteComparison', symbol);
    // set compared property to false to hide the comparison, so it can be enabled again if the same comparison is added
    for (var i = 0; i < chart.dataSets.length; i++) {
      if (chart.dataSets[i].title == symbol) {
        chart.dataSets[i].compared = false;
      }
    }
    chart.validateData();
  }  
  
  $comparisonSelect.initCoinsSearchAutocomplete(TEXT.compare, (screenWidth<=1200?300:440) + 'px');

  $comparisonSelect.on('change', function() {
    var $select = $(this);
    var symbols = $select.val();
    var removedSymbols = comparisonSelectSymbols.subtract(symbols);
    var addedSymbols = symbols ? symbols.subtract(comparisonSelectSymbols) : [];
    comparisonSelectSymbols = symbols;

    if (removedSymbols.length)
      deleteComparison(removedSymbols[0]);

    if (addedSymbols.length)
      addComparison(addedSymbols[0]);
  });

  $addFavoriteButton.on('click', function () {
    $addFavoriteButton.addClass('disabled');
    $.ajax({
      url: API.addFavorite + '/' + COIN_SYMBOL,
      dataType: 'json',
      async: true,
      cache: false,
      success: function (response) {
        log('Add to favorites', response);
        if (response.success) {
          $addFavoriteButton.removeClass('disabled');
          $addFavoriteButton.addClass('hide');
          $removeFavoriteButton.removeClass('hide');
        }
      },
      error: handleAjaxError
    });
  });

  $removeFavoriteButton.on('click', function () {
    $removeFavoriteButton.addClass('disabled');
    $.ajax({
      url: API.removeFavorite + '/' + COIN_SYMBOL,
      dataType: 'json',
      async: true,
      cache: false,
      success: function (response) {
        log('Remove from favorites', response);
        if (response.success) {
          $removeFavoriteButton.removeClass('disabled');
          $removeFavoriteButton.addClass('hide');
          $addFavoriteButton.removeClass('hide');
        }
      },
      error: handleAjaxError
    });
  });

  $('#copy-widget-code-button').on('click', function () {
    var $input = $('#widget-code');
    $input.copyToClipboard();
  });
  
  window.reinitChart = function() {
    chart.clear();
    chart = buildChart();
  };
  
});